<?php
// ArriereController.php — VERSION FINALE 2025
// Génération correcte des arriérés : du MOIS SUIVANT jusqu'à AUJOURD'HUI
// Règles : avant 2024 => 200 FCFA / 2024 et + => 250 FCFA
// Réponses JSON propres pour Flutter

require_once __DIR__ . '/../config/database.php';

class ArriereController {
    private $db;

    public function __construct() {
        $database = new Database();
        $this->db = $database->getConnection();

        header("Access-Control-Allow-Origin: *");
        header("Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS");
        header("Access-Control-Allow-Headers: Content-Type, Authorization");
        header("Content-Type: application/json; charset=utf-8");

        if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
            http_response_code(200);
            exit;
        }
    }

    /* ============================
       GET /api/arrieres?action=getAll
       ============================ */
    public function getAll() {
        try {
            $sql = $this->db->prepare("
                SELECT a.*, m.nom, m.prenom
                FROM historiques_arriere a
                LEFT JOIN membres m ON m.id = a.membre_id
                ORDER BY a.annee DESC, a.mois DESC
            ");
            $sql->execute();
            $res = $sql->fetchAll(PDO::FETCH_ASSOC);
            $this->respond(true, ["data" => $res]);
        } catch (Exception $e) {
            $this->respond(false, ["message" => "Erreur serveur : ".$e->getMessage()]);
        }
    }

    /* ============================
       DELETE /api/arrieres?action=delete&id=XX
       ============================ */
    public function delete($id) {
        $id = (int)$id;
        if ($id <= 0) {
            $this->respond(false, ["message" => "ID invalide"]);
            return;
        }

        try {
            $stmt = $this->db->prepare("DELETE FROM historiques_arriere WHERE id=?");
            $ok = $stmt->execute([$id]);
            $this->respond($ok, ["message" => $ok ? "Arriéré supprimé." : "Erreur suppression"]);
        } catch (Exception $e) {
            $this->respond(false, ["message" => $e->getMessage()]);
        }
    }

    /* =====================================================
       POST /api/arrieres?action=add
       LOGIQUE :
       1. startMonth/startYear = mois indiqué PAR L'ADMIN
       2. Premier arriéré = MOIS SUIVANT
       3. On génère jusqu'au mois courant
    ====================================================== */
    public function add() {
        $input = $this->getJsonInput();

        if (empty($input["membre_id"]) || empty($input["mois"]) || empty($input["annee"])) {
            $this->respond(false, ["message" => "Paramètres manquants"]);
            return;
        }

        $membreId   = (int)$input["membre_id"];
        $startMonth = (int)$input["mois"];
        $startYear  = (int)$input["annee"];

        if ($startMonth < 1 || $startMonth > 12) {
            $this->respond(false, ["message" => "Mois invalide"]);
            return;
        }

        /* 🔥 Correction fondamentale :
           Si l'admin met DECEMBRE 2024,
           le premier mois d'arriéré = JANVIER 2025
        */
        $startMonth++;
        if ($startMonth == 13) {
            $startMonth = 1;
            $startYear++;
        }

        $currentYear  = (int)date("Y");
        $currentMonth = (int)date("n");

        $totalAdded = 0;
        $details = [];

        try {
            $this->db->beginTransaction();

            $year = $startYear;
            $month = $startMonth;

            while (true) {
                if ($year > $currentYear ||
                    ($year == $currentYear && $month > $currentMonth)) {
                    break;
                }

                if (!$this->arriereExists($membreId, $year, $month)) {

                    $montantDu = $this->determineMontantDu($year);

                    $stmt = $this->db->prepare("
                        INSERT INTO historiques_arriere
                        (membre_id, annee, mois, montant_du, montant_paye, statut, created_at)
                        VALUES (?, ?, ?, ?, 0, 'impaye', NOW())
                    ");
                    $stmt->execute([$membreId, $year, $month, $montantDu]);

                    $totalAdded += $montantDu;
                    $details[] = [
                        "annee" => $year,
                        "mois" => $month,
                        "montant_du" => $montantDu
                    ];
                }

                // next month
                $month++;
                if ($month > 12) {
                    $month = 1;
                    $year++;
                }
            }

            $this->db->commit();

            $this->respond(true, [
                "message" => "Arriérés générés",
                "total_added" => $totalAdded,
                "details" => $details
            ]);

        } catch (Exception $e) {
            $this->db->rollBack();
            $this->respond(false, ["message" => $e->getMessage()]);
        }
    }

    /* ============================
       POST /api/arrieres?action=update
       ============================ */
    public function update() {
        $input = $this->getJsonInput();

        if (empty($input["id"]) || empty($input["membre_id"]) ||
            !isset($input["mois"]) || !isset($input["annee"])) {
            $this->respond(false, ["message" => "Paramètres manquants"]);
            return;
        }

        $id = (int)$input["id"];
        $membreId = (int)$input["membre_id"];
        $mois = (int)$input["mois"];
        $annee = (int)$input["annee"];
        $montantDu = $this->determineMontantDu($annee);
        $statut = $input["statut"] ?? "impaye";
        $montantPaye = isset($input["montant_paye"]) ? (float)$input["montant_paye"] : 0;

        try {
            $stmt = $this->db->prepare("
                UPDATE historiques_arriere
                SET membre_id=?, annee=?, mois=?, montant_du=?, montant_paye=?, statut=?
                WHERE id=?
            ");
            $ok = $stmt->execute([
                $membreId, $annee, $mois, $montantDu, $montantPaye, $statut, $id
            ]);

            $this->respond($ok, [
                "message" => $ok ? "Arriéré mis à jour" : "Erreur mise à jour",
                "montant_du" => $montantDu
            ]);

        } catch (Exception $e) {
            $this->respond(false, ["message" => $e->getMessage()]);
        }
    }

    /* ============================
       REGULARISATION (version stable)
       ============================ */
    public function regulariser() {
        $this->respond(true, ["message" => "Régularisation – OK (version simple)"]);
    }

    /* ============================
       UTILITAIRES
       ============================ */
    private function determineMontantDu($annee) {
        return ($annee < 2024) ? 200.0 : 250.0;
    }

    private function arriereExists($membreId, $annee, $mois) {
        $stmt = $this->db->prepare("
            SELECT COUNT(*) FROM historiques_arriere
            WHERE membre_id=? AND annee=? AND mois=?
        ");
        $stmt->execute([$membreId, $annee, $mois]);
        return $stmt->fetchColumn() > 0;
    }

    private function getJsonInput() {
        return json_decode(file_get_contents("php://input"), true) ?: [];
    }

    private function respond($success, $data = []) {
        echo json_encode(array_merge(["success" => $success], $data), JSON_UNESCAPED_UNICODE);
        exit;
    }
}

/* ============================
   DISPATCHER
   ============================ */
$controller = new ArriereController();
$action = $_GET["action"] ?? "";

switch ($action) {
    case "getAll":     $controller->getAll(); break;
    case "add":        $controller->add(); break;
    case "update":     $controller->update(); break;
    case "delete":     $controller->delete($_GET["id"] ?? 0); break;
    case "regulariser":$controller->regulariser(); break;
    default:
        echo json_encode(["success" => false, "message" => "Action invalide"]);
        break;
}
